/*-
 * Copyright (C)2012..2024 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2012..2024 @BABOLO http://www.babolo.ru/\n"
#ident "@(#) $Id: octets.c,v 1.22 2024/01/08 00:11:11 babolo Exp $"

#define MIFE_COMPAT     5
#define BLIN_COMPAT     4
#define RECOBE_COMPAT   VMAJOR
#define RECOBE_INTERNAL 1

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <babolo/BLINflag.h>
#include <mife.h>
#include "recobe.h"

typedef union recobe_udes {                                 /*     UTF-8    */
    u_int16_t all;
    struct {
        u_int16_t value  :  8
                , shift  :  4
                , dummy  :  2
                , next   :  1
                , first  :  1
        ;
    };
} recobe_udes;

typedef union recobe_sdes {                                 /*     CLEN     */
    u_int16_t all;
    struct {
        u_int16_t val    :  8
                , dummy  :  4
                , len    :  4
        ;
    };
} recobe_cdes;

const u_int8_t
/*****************************************************************************************************
 *****************************************************************************************************
 ****                                                                       ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_cnt1[256] =                                                                               /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
{ 0, 1, 1, 2,  1, 2, 2, 3,  1, 2, 2, 3,  2, 3, 3, 4
, 1, 2, 2, 3,  2, 3, 3, 4,  2, 3, 3, 4,  3, 4, 4, 5
, 1, 2, 2, 3,  2, 3, 3, 4,  2, 3, 3, 4,  3, 4, 4, 5
, 2, 3, 3, 4,  3, 4, 4, 5,  3, 4, 4, 5,  4, 5, 5, 6

, 1, 2, 2, 3,  2, 3, 3, 4,  2, 3, 3, 4,  3, 4, 4, 5
, 2, 3, 3, 4,  3, 4, 4, 5,  3, 4, 4, 5,  4, 5, 5, 6
, 2, 3, 3, 4,  3, 4, 4, 5,  3, 4, 4, 5,  4, 5, 5, 6
, 3, 4, 4, 5,  4, 5, 5, 6,  4, 5, 5, 6,  5, 6, 6, 7

, 1, 2, 2, 3,  2, 3, 3, 4,  2, 3, 3, 4,  3, 4, 4, 5
, 2, 3, 3, 4,  3, 4, 4, 5,  3, 4, 4, 5,  4, 5, 5, 6
, 2, 3, 3, 4,  3, 4, 4, 5,  3, 4, 4, 5,  4, 5, 5, 6
, 3, 4, 4, 5,  4, 5, 5, 6,  4, 5, 5, 6,  5, 6, 6, 7

, 2, 3, 3, 4,  3, 4, 4, 5,  3, 4, 4, 5,  4, 5, 5, 6
, 3, 4, 4, 5,  4, 5, 5, 6,  4, 5, 5, 6,  5, 6, 6, 7
, 3, 4, 4, 5,  4, 5, 5, 6,  4, 5, 5, 6,  5, 6, 6, 7
, 4, 5, 5, 6,  5, 6, 6, 7,  5, 6, 6, 7,  6, 7, 7, 8
};

u_int64_t
/*****************************************************************************************************
 *****************************************************************************************************
 ****   octets len   c LSB  -len MSB                                        ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_load(const u_char *octets, ssize_t len) {                                                 /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int64_t r;
    ssize_t   i;

    r = 0;
    if  (len < 0) {
        for (i = 0; i < -len; ++i) {
            r <<= 8;
            r |= octets[i];
        }
    } else {
        for (i = len; i > 0;) {
            r <<= 8;
            r |= octets[--i];
    }   }
    return(r);
}

void
/*****************************************************************************************************
 *****************************************************************************************************
 ****   octets len    c LSB  -len MSB                                 ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_store(u_char *octets, ssize_t len, u_int64_t c) {                                         /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    ssize_t i;

    if  (len < 0) {
        for (i = 0; i > len; --i) {
            octets[i - len - 1] = c & 0xFF;
            c >>= 8;
        }
    } else {
        for (i = 0; i < len; ++i) {
            octets[i] = c & 0xFF;
            c >>= 8;
}   }   }

static const recobe_udes
/*****************************************************************************************************/
recobe_ud[256] =                                                                                 /****
 *****************************************************************************************************/
{ {0x8000}, {0x8001}, {0x8002}, {0x8003}, {0x8004}, {0x8005}, {0x8006}, {0x8007} /* 00 */
, {0x8008}, {0x8009}, {0x800A}, {0x800B}, {0x800C}, {0x800D}, {0x800E}, {0x800F} /* 08 */
, {0x8010}, {0x8011}, {0x8012}, {0x8013}, {0x8014}, {0x8015}, {0x8016}, {0x8017} /* 10 */
, {0x8018}, {0x8019}, {0x801A}, {0x801B}, {0x801C}, {0x801D}, {0x801E}, {0x801F} /* 18 */
, {0x8020}, {0x8021}, {0x8022}, {0x8023}, {0x8024}, {0x8025}, {0x8026}, {0x8027} /* 20 */
, {0x8028}, {0x8029}, {0x802A}, {0x802B}, {0x802C}, {0x802D}, {0x802E}, {0x802F} /* 28 */
, {0x8030}, {0x8031}, {0x8032}, {0x8033}, {0x8034}, {0x8035}, {0x8036}, {0x8037} /* 30 */
, {0x8038}, {0x8039}, {0x803A}, {0x803B}, {0x803C}, {0x803D}, {0x803E}, {0x803F} /* 38 */
, {0x8040}, {0x8041}, {0x8042}, {0x8043}, {0x8044}, {0x8045}, {0x8046}, {0x8047} /* 40 */
, {0x8048}, {0x8049}, {0x804A}, {0x804B}, {0x804C}, {0x804D}, {0x804E}, {0x804F} /* 48 */
, {0x8050}, {0x8051}, {0x8052}, {0x8053}, {0x8054}, {0x8055}, {0x8056}, {0x8057} /* 50 */
, {0x8058}, {0x8059}, {0x805A}, {0x805B}, {0x805C}, {0x805D}, {0x805E}, {0x805F} /* 58 */
, {0x8060}, {0x8061}, {0x8062}, {0x8063}, {0x8064}, {0x8065}, {0x8066}, {0x8067} /* 60 */
, {0x8068}, {0x8069}, {0x806A}, {0x806B}, {0x806C}, {0x806D}, {0x806E}, {0x806F} /* 68 */
, {0x8070}, {0x8071}, {0x8072}, {0x8073}, {0x8074}, {0x8075}, {0x8076}, {0x8077} /* 70 */
, {0x8078}, {0x8079}, {0x807A}, {0x807B}, {0x807C}, {0x807D}, {0x807E}, {0x807F} /* 78 */
, {0x4000}, {0x4001}, {0x4002}, {0x4003}, {0x4004}, {0x4005}, {0x4006}, {0x4007} /* 80 */
, {0x4008}, {0x4009}, {0x400A}, {0x400B}, {0x400C}, {0x400D}, {0x400E}, {0x400F} /* 88 */
, {0x4010}, {0x4011}, {0x4012}, {0x4013}, {0x4014}, {0x4015}, {0x4016}, {0x4017} /* 90 */
, {0x4018}, {0x4019}, {0x401A}, {0x401B}, {0x401C}, {0x401D}, {0x401E}, {0x401F} /* 98 */
, {0x4020}, {0x4021}, {0x4022}, {0x4023}, {0x4024}, {0x4025}, {0x4026}, {0x4027} /* A0 */
, {0x4028}, {0x4029}, {0x402A}, {0x402B}, {0x402C}, {0x402D}, {0x402E}, {0x402F} /* A8 */
, {0x4030}, {0x4031}, {0x4032}, {0x4033}, {0x4034}, {0x4035}, {0x4036}, {0x4037} /* B0 */
, {0x4038}, {0x4039}, {0x403A}, {0x403B}, {0x403C}, {0x403D}, {0x403E}, {0x403F} /* B8 */
, {0x8100}, {0x8101}, {0x8102}, {0x8103}, {0x8104}, {0x8105}, {0x8106}, {0x8107} /* C0 */
, {0x8108}, {0x8109}, {0x810A}, {0x810B}, {0x810C}, {0x810D}, {0x810E}, {0x810F} /* C8 */
, {0x8110}, {0x8111}, {0x8112}, {0x8113}, {0x8114}, {0x8115}, {0x8116}, {0x8117} /* D0 */
, {0x8118}, {0x8119}, {0x811A}, {0x811B}, {0x811C}, {0x811D}, {0x811E}, {0x811F} /* D8 */
, {0x8200}, {0x8201}, {0x8202}, {0x8203}, {0x8204}, {0x8205}, {0x8206}, {0x8207} /* E0 */
, {0x8208}, {0x8209}, {0x820A}, {0x820B}, {0x820C}, {0x820D}, {0x820E}, {0x820F} /* E8 */
, {0x8300}, {0x8301}, {0x8302}, {0x8303}, {0x8304}, {0x8305}, {0x8306}, {0x8307} /* F0 */
, {0x8400}, {0x8401}, {0x8402}, {0x8403}, {0x8500}, {0x8501}, {0x8600}, {0x0000} /* F8 */
};

u_int64_t
/*****************************************************************************************************
 *****************************************************************************************************
 ****   utf8   octets ,    len .             ****
 ****     len,   -  .             ****
 ****   len .                                                               ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_uload(const u_char *octets, ssize_t *len) {                                               /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    ssize_t       inpos = 0;
    u_int64_t     r = 0;
    recobe_udes   ds;

    for (int i = 1, j = 0; i > 0; --i, ++j) {
        if  (inpos >= *len) {
            inpos = ~inpos;
            goto out;
        }
        ds = recobe_ud[octets[inpos++]];
        if  ((!j) ? !ds.first : !ds.next) {
            inpos = ~inpos;
            goto out;
        }
        if  (!!ds.first) i = ds.shift + 1;
        r |= (u_int64_t)ds.value << ((i - 1) * 6);
    }
out:;
    *len = inpos;
    return(r);
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****   utf8   octet                                                     ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_ulen(u_char octet) {                                                                      /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    return(recobe_ud[octet].first ? recobe_ud[octet].shift + 1 : -1);
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****   octets utf8  c,                        ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_ustore(u_char *octets, u_int64_t c) {                                                     /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    int       r = 0;
    u_int64_t m = 0x0000000000000080ULL;

    for (int i = 0; i < 12; ++i) {
        if  ((m > c) || (12 == i)) {
            if  (!i) {
                octets[r++] = (u_char)c;
            } else {
                octets[r++] = (u_char)((0x00FFU << (7 - i)) | (c >> (6 * i)));
                for (; i > 0;) {
                    octets[r++] = 0x0080 | (0x003F & (c >> (6 * --i)));
            }   }
            goto out;
        }
        m <<= !i ? 4 : 5;
    }
    r = -1;
out:
    return(r);
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****   utf8   c                                                    ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_utest(u_int64_t c) {                                                                      /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    int r;

    c >>= 3;
    for (r = 0; r < 13; ++r) if (!(c >>= ((2 > r) ? 4 : 5))) goto out;
    r = -1;
out:
    return(++r);
}

static const recobe_cdes
/*****************************************************************************************************/
recobe_cd[256] =                                                                                 /****
 *****************************************************************************************************/
{ {0x0000}, {0x0001}, {0x0002}, {0x0003}, {0x0004}, {0x0005}, {0x0006}, {0x0007} /* 00 */
, {0x0008}, {0x0009}, {0x000A}, {0x000B}, {0x000C}, {0x000D}, {0x000E}, {0x000F} /* 08 */
, {0x0010}, {0x0011}, {0x0012}, {0x0013}, {0x0014}, {0x0015}, {0x0016}, {0x0017} /* 10 */
, {0x0018}, {0x0019}, {0x001A}, {0x001B}, {0x001C}, {0x001D}, {0x001E}, {0x001F} /* 18 */
, {0x0020}, {0x0021}, {0x0022}, {0x0023}, {0x0024}, {0x0025}, {0x0026}, {0x0027} /* 20 */
, {0x0028}, {0x0029}, {0x002A}, {0x002B}, {0x002C}, {0x002D}, {0x002E}, {0x002F} /* 28 */
, {0x0030}, {0x0031}, {0x0032}, {0x0033}, {0x0034}, {0x0035}, {0x0036}, {0x0037} /* 30 */
, {0x0038}, {0x0039}, {0x003A}, {0x003B}, {0x003C}, {0x003D}, {0x003E}, {0x003F} /* 38 */
, {0x0040}, {0x0041}, {0x0042}, {0x0043}, {0x0044}, {0x0045}, {0x0046}, {0x0047} /* 40 */
, {0x0048}, {0x0049}, {0x004A}, {0x004B}, {0x004C}, {0x004D}, {0x004E}, {0x004F} /* 48 */
, {0x0050}, {0x0051}, {0x0052}, {0x0053}, {0x0054}, {0x0055}, {0x0056}, {0x0057} /* 50 */
, {0x0058}, {0x0059}, {0x005A}, {0x005B}, {0x005C}, {0x005D}, {0x005E}, {0x005F} /* 58 */
, {0x0060}, {0x0061}, {0x0062}, {0x0063}, {0x0064}, {0x0065}, {0x0066}, {0x0067} /* 60 */
, {0x0068}, {0x0069}, {0x006A}, {0x006B}, {0x006C}, {0x006D}, {0x006E}, {0x006F} /* 68 */
, {0x0070}, {0x0071}, {0x0072}, {0x0073}, {0x0074}, {0x0075}, {0x0076}, {0x0077} /* 70 */
, {0x0078}, {0x0079}, {0x007A}, {0x007B}, {0x007C}, {0x007D}, {0x007E}, {0x007F} /* 78 */
, {0x1000}, {0x1001}, {0x1002}, {0x1003}, {0x1004}, {0x1005}, {0x1006}, {0x1007} /* 80 */
, {0x1008}, {0x1009}, {0x100A}, {0x100B}, {0x100C}, {0x100D}, {0x100E}, {0x100F} /* 88 */
, {0x1010}, {0x1011}, {0x1012}, {0x1013}, {0x1014}, {0x1015}, {0x1016}, {0x1017} /* 90 */
, {0x1018}, {0x1019}, {0x101A}, {0x101B}, {0x101C}, {0x101D}, {0x101E}, {0x101F} /* 98 */
, {0x1020}, {0x1021}, {0x1022}, {0x1023}, {0x1024}, {0x1025}, {0x1026}, {0x1027} /* A0 */
, {0x1028}, {0x1029}, {0x102A}, {0x102B}, {0x102C}, {0x102D}, {0x102E}, {0x102F} /* A8 */
, {0x1030}, {0x1031}, {0x1032}, {0x1033}, {0x1034}, {0x1035}, {0x1036}, {0x1037} /* B0 */
, {0x1038}, {0x1039}, {0x103A}, {0x103B}, {0x103C}, {0x103D}, {0x103E}, {0x103F} /* B8 */
, {0x2000}, {0x2001}, {0x2002}, {0x2003}, {0x2004}, {0x2005}, {0x2006}, {0x2007} /* C0 */
, {0x2008}, {0x2009}, {0x200A}, {0x200B}, {0x200C}, {0x200D}, {0x200E}, {0x200F} /* C8 */
, {0x2010}, {0x2011}, {0x2012}, {0x2013}, {0x2014}, {0x2015}, {0x2016}, {0x2017} /* D0 */
, {0x2018}, {0x2019}, {0x201A}, {0x201B}, {0x201C}, {0x201D}, {0x201E}, {0x201F} /* D8 */
, {0x3000}, {0x3001}, {0x3002}, {0x3003}, {0x3004}, {0x3005}, {0x3006}, {0x3007} /* E0 */
, {0x3008}, {0x3009}, {0x300A}, {0x300B}, {0x300C}, {0x300D}, {0x300E}, {0x300F} /* E8 */
, {0x4000}, {0x4001}, {0x4002}, {0x4003}, {0x4004}, {0x4005}, {0x4006}, {0x4007} /* F0 */
, {0x5000}, {0x5001}, {0x5002}, {0x5003}, {0x6000}, {0x6001}, {0x7000}, {0x8000} /* F8 */
};

u_int64_t
/*****************************************************************************************************
 *****************************************************************************************************
 ****   octets    ,    len .           ****
 ****     len,   -  .             ****
 ****   len .                                                               ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_cload(const u_char *octets, ssize_t *len) {                                               /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    ssize_t       inpos = 0;
    u_int64_t     r = 0;
    recobe_cdes   ds;

    for (int i = 1; i >= 0; --i, ++inpos) {
        if  (inpos >= *len) {
            inpos = ~inpos;
            goto out;
        }
        ds = recobe_cd[octets[inpos]];
        if  (!inpos) {
            i = ds.len;
            r = (u_int64_t)ds.val << (8 * i);
        } else {
            r |= (u_int64_t)octets[inpos] << (i * 8);
    }   }
out:;
    *len = inpos;
    return(r);
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****      c                                                ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_clen(u_char octet) {                                                                      /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    return(recobe_cd[octet].len + 1);
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****   octets    ,    len .           ****
 ****     len,   -  .             ****
 ****   len .                                                               ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_cstore(u_char *octets, u_int64_t c) {                                                     /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    int       r;
    int       i;

    for (i = 0; i < 8; ++i) if  ((0x080ULL << (7 * i)) > c) break;
    for (r = 0; i >= 0; --i) {
        if  (!r) {
            octets[r++] = (u_char)((c >> (8 * i)) | (0x00FFU << (8 - i)));
        } else {
            octets[r++] = (c >> (8 * i)) & 0x00FF;
    }   }
    return(r);
}

u_int64_t
/*****************************************************************************************************
 *****************************************************************************************************
 ****    mask.                                                                     ****
 ****   mask ,        octets    ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_loadM(const u_char *octets, const char *mask) {                                           /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int64_t r = 0;
    ssize_t   i;

    for (i = 0; !!mask[i]; ++i) {
        if  ((mask[i] < '0') || (mask[i] > '7')) {
            ifBLIN_QX0("Illegal input %c in %s[%zd]", mask[i], mask, i);
            errno = EINVAL;
            break;
        }
        r |= (u_int64_t)octets[i] << ((mask[i] - '0') << 3);
    }
    return(r);
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****   octets   c   mask.                                                 ****
 ****   mask ,   c      octets           ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_storeM(u_char *octets, const char *mask, u_int64_t c) {                                   /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    int i;

    for (i = 0; !!mask[i]; ++i) {
        if  ((mask[i] < '0') || (mask[i] > '7')) {
            ifBLIN_QX0("Illegal input %c in %s[%d]", mask[i], mask, i);
            i = -1;
            errno = EINVAL;
            break;
        }
        octets[i] = (u_char)(c >> ((mask[i] - '0') << 3));
    }
    return(i);
}

u_int64_t
/*****************************************************************************************************
 *****************************************************************************************************
 ****      c   m                                      ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_swap(u_int64_t c, u_int8_t m) {                                                           /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    if  (m & 0x01) c = ((c >>  1) & 0x5555555555555555ULL) | ((c <<  1) & 0xAAAAAAAAAAAAAAAAULL);
    if  (m & 0x02) c = ((c >>  2) & 0x3333333333333333ULL) | ((c <<  2) & 0xCCCCCCCCCCCCCCCCULL);
    if  (m & 0x04) c = ((c >>  4) & 0x0F0F0F0F0F0F0F0FULL) | ((c <<  4) & 0xF0F0F0F0F0F0F0F0ULL);
    if  (m & 0x08) c = ((c >>  8) & 0x00FF00FF00FF00FFULL) | ((c <<  8) & 0xFF00FF00FF00FF00ULL);
    if  (m & 0x10) c = ((c >> 16) & 0x0000FFFF0000FFFFULL) | ((c << 16) & 0xFFFF0000FFFF0000ULL);
    if  (m & 0x20) c = ((c >> 32) & 0x00000000FFFFFFFFULL) | ((c << 32) & 0xFFFFFFFF00000000ULL);
    return(c);
}

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****     c                                                            ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_count(u_int64_t c) {                                                                      /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    int i;

    for (i = 0; !!c; c >>= 8) {
        i += recobe_cnt1[c & 0xFF];
    }
    return(i);
}

const u_char
/*****************************************************************************************************
 *****************************************************************************************************
 ****                                                                    ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_most1[256] =                                                                              /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
{ 0x00, 0x01, 0x02, 0x02, 0x03, 0x03, 0x03, 0x03, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04
, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05
, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06
, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06
, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07
, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07
, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07
, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07
, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08
, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08
, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08
, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08
, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08
, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08
, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08
, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08
};

int
/*****************************************************************************************************
 *****************************************************************************************************
 ****     c                                                                   ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_wide(u_int64_t c) {                                                                       /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    int i;
    int k;

    for (i = 0, k = 0; !!c; c >>= 8, k += 8) {
        i = recobe_most1[c & 0x0FF] + k;
    }
    return(i);
}
