/*-
 * Copyright (C)2012..2024 @BABOLO http://www.babolo.ru/
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ident "@(#) Copyright (C)2012..2024 @BABOLO http://www.babolo.ru/\n"
#ident "@(#) $Id: crc.c,v 1.14 2024/07/28 20:09:54 babolo Exp $"

#define MIFE_COMPAT     5
#define BLIN_COMPAT     4
#define Bpars_COMPAT    4
#define RECOBE_COMPAT   VMAJOR
#define RECOBE_INTERNAL 1

#include <sys/types.h>
#include <sys/event.h>
#include <sys/time.h>
#include <sysexits.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <babolo/BLINflag.h>
#include <babolo/parser.h>
#include <mife.h>
#include "recobe.h"
#include "crc.h"

#define CRC_L 6

#define CRC32_INIT  0xFFFFFFFF
#define CRC32_POLY  0xEDB88320
#define CRC24_INIT  0x00B704CE
#define CRC24_POLY  0x01864CFB

/*****************************************************************************************************
 *****************************************************************************************************/
static const char *                                                                              /****/
crc_1[] = {"CSUM1"      , NULL};                                                                 /****/
                                                                                                 /****/
static const char *                                                                              /****/
crc_24[] = {"CRC24"     , NULL};                                                                 /****/
                                                                                                 /****/
static const char *                                                                              /****/
crc_32[] = {"CRC32"     , NULL};                                                                 /****/
                                                                                                 /****/
static const char *                                                                              /****/
crc_itt[] = {"CCITT"    , NULL};                                                                 /****/
                                                                                                 /****/
static const char *                                                                              /****/
crc_csum[] = {"CSUM"    , NULL};                                                                 /****/
/*****************************************************************************************************
 *****************************************************************************************************/

static const recobe_c_tnm
/*****************************************************************************************************
                                                                                                  ****/
encolist[CRC_L] =                                                                                /****
                                                                                                  ****
 *****************************************************************************************************/
{ { NULL                , NULL     , NULL}
, { (const u_char *)"1" , &crc_1   , NULL}
, { (const u_char *)"2" , &crc_24  , NULL}
, { (const u_char *)"3" , &crc_32  , NULL}
, { (const u_char *)"4" , &crc_itt , NULL}
, { (const u_char *)"5" , &crc_csum, NULL}
};

static u_int
/*****************************************************************************************************
                                                                                                  ****/
codnumbyid(const u_char *nm) {                                                                   /****
                                                                                                  ****
 *****************************************************************************************************/
    return(babolo_testword(&crc, (const char *)nm));
}

static recobe_conv
/*****************************************************************************************************
                                                                                                  ****/
convbyname(const char *name) {                                                                   /****
                                                                                                  ****
 *****************************************************************************************************/
    recobe_conv cnv = {0, 0, 0, 0, 0};

    if  (!name || !name[0]) {
        cnv.nonesrc = cnv.nonedst = 1;
    } else {
        cnv.encodst = codnumbyid((const u_char *)name);
    }
    cnv.domain = RECOBE_CRC;
    return(cnv);
}

u_int32_t                     /* IEEE 802.3 */
/*****************************************************************************************************
 *****************************************************************************************************
                                                                                                  ****/
recobe_crc32(const u_char *octets, ssize_t len) {                                                /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int32_t carry;
    u_int32_t r;
    ssize_t   i;
    int       j;

    r = CRC32_INIT;
    if  (len < 0) {
        for (i = -len; i > 0;) {
            r ^= octets[--i];
            for (j = 0; j < 8; ++j) {
                carry = r & 1;
                r >>= 1;
                if  (carry) r ^= CRC32_POLY;
        }   }
    } else {
        for (i = 0; i < len; ++i) {
            r ^= octets[i];
            for (j = 0; j < 8; ++j) {
                carry = r & 1;
                r >>= 1;
                if  (carry) r ^= CRC32_POLY;
    }   }   }
    r ^= CRC32_INIT;
    return(r);
}

static int
/*****************************************************************************************************
                                                                                                  ****/
crc32(recobe_chain *chain, recobe_chunk *chunk) {                                                /****
                                                                                                  ****
 *****************************************************************************************************/
#   define blin_internal_flags (chain->flags)
    ssize_t       intail;
    u_int32_t     carry;
    u_int64_t     wchr;
    int           ex = EX_OK;
    u_int32_t    *r;
    int           j;

    ifBLIN_QX3("+ %08X %"BLIN_X, chain->flags, BLIN_I(chain));
    intail = (ssize_t)recobe_chunksize(chain->tail);
    if  (!(chain->flags & RECOBE_USED)) {
        r = (u_int32_t *)&chain->converdata;
        *r = CRC32_INIT;
        chain->flags |= RECOBE_DOEND | RECOBE_USED;
    } else if (!(chain->flags & RECOBE_DOEND)) {
        chunk->position = RECOBE_EOT;
        goto out;
    }
    r = (u_int32_t *)&chain->converdata;
    if  (intail < 1) {
        *r ^= CRC32_INIT;
        if  (0 > (ex = recobe_outword(chain, chunk, *r))) ifBLIN_QW0("out");
        chain->flags &= ~RECOBE_DOEND;
        goto out;
    } else while(0 < recobe_inword(chain, chain->tail, &wchr)) {
        *r ^= wchr;
        for (j = 0; j < 8; ++j) {
            carry = *r & 1;
            *r >>= 1;
            if  (carry) *r ^= CRC32_POLY;
    }   }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

u_int32_t                       /* RFC-2440 */
/*****************************************************************************************************
 *****************************************************************************************************
                                                                                                  ****/
recobe_crc24(const u_char *octets, ssize_t len) {                                                /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int32_t r;
    ssize_t   i;
    int       j;

    r = CRC24_INIT;
    if  (len < 0) {
        for (i = -len; i > 0;) {
            r ^= (u_int32_t)octets[--i] << 16;
            for (j = 0; j < 8; ++j)  {
                r <<= 1;
                if  (r & 0x1000000) r ^= CRC24_POLY;
        }   }
    } else {
        for (i = 0; i < len; ++i) {
            r ^= (u_int32_t)octets[i] << 16;
            for (j = 0; j < 8; ++j)  {
                r <<= 1;
                if  (r & 0x1000000) r ^= CRC24_POLY;
    }   }   }
    r &= 0x00FFFFFF;
    return(r);
}

static int
/*****************************************************************************************************
                                                                                                  ****/
crc24(recobe_chain *chain, recobe_chunk *chunk) {                                                /****
                                                                                                  ****
 *****************************************************************************************************/
#   define blin_internal_flags (chain->flags)
    ssize_t       intail;
    u_int64_t     wchr;
    int           ex = EX_OK;
    u_int32_t    *r;
    int           j;

    ifBLIN_QX3("+ %08X %"BLIN_X, chain->flags, BLIN_I(chain));
    intail = (ssize_t)recobe_chunksize(chain->tail);
    if  (!(chain->flags & RECOBE_USED)) {
        r = (u_int32_t *)&chain->converdata;
        *r = CRC24_INIT;
        chain->flags |= RECOBE_DOEND | RECOBE_USED;
    } else if (!(chain->flags & RECOBE_DOEND)) {
        chunk->position = RECOBE_EOT;
        goto out;
    }
    r = (u_int32_t *)&chain->converdata;
    if  (intail < 1) {
        *r &= 0x00FFFFFF;
        if  (0 > (ex = recobe_outword(chain, chunk, *r))) ifBLIN_QW0("out");
        chain->flags &= ~RECOBE_DOEND;
    } else while(0 < recobe_inword(chain, chain->tail, &wchr)) {
        *r ^= wchr << 16;
        for (j = 0; j < 8; ++j)  {
            *r <<= 1;
            if  (*r & 0x1000000) *r ^= CRC24_POLY;
    }   }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static const u_int16_t crccitt[256] =
{ 0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50A5, 0x60C6, 0x70E7
, 0x8108, 0x9129, 0xA14A, 0xB16B, 0xC18C, 0xD1AD, 0xE1CE, 0xF1EF
, 0x1231, 0x0210, 0x3273, 0x2252, 0x52B5, 0x4294, 0x72F7, 0x62D6
, 0x9339, 0x8318, 0xB37B, 0xA35A, 0xD3BD, 0xC39C, 0xF3FF, 0xE3DE
, 0x2462, 0x3443, 0x0420, 0x1401, 0x64E6, 0x74C7, 0x44A4, 0x5485
, 0xA56A, 0xB54B, 0x8528, 0x9509, 0xE5EE, 0xF5CF, 0xC5AC, 0xD58D
, 0x3653, 0x2672, 0x1611, 0x0630, 0x76D7, 0x66F6, 0x5695, 0x46B4
, 0xB75B, 0xA77A, 0x9719, 0x8738, 0xF7DF, 0xE7FE, 0xD79D, 0xC7BC
, 0x48C4, 0x58E5, 0x6886, 0x78A7, 0x0840, 0x1861, 0x2802, 0x3823
, 0xC9CC, 0xD9ED, 0xE98E, 0xF9AF, 0x8948, 0x9969, 0xA90A, 0xB92B
, 0x5AF5, 0x4AD4, 0x7AB7, 0x6A96, 0x1A71, 0x0A50, 0x3A33, 0x2A12
, 0xDBFD, 0xCBDC, 0xFBBF, 0xEB9E, 0x9B79, 0x8B58, 0xBB3B, 0xAB1A
, 0x6CA6, 0x7C87, 0x4CE4, 0x5CC5, 0x2C22, 0x3C03, 0x0C60, 0x1C41
, 0xEDAE, 0xFD8F, 0xCDEC, 0xDDCD, 0xAD2A, 0xBD0B, 0x8D68, 0x9D49
, 0x7E97, 0x6EB6, 0x5ED5, 0x4EF4, 0x3E13, 0x2E32, 0x1E51, 0x0E70
, 0xFF9F, 0xEFBE, 0xDFDD, 0xCFFC, 0xBF1B, 0xAF3A, 0x9F59, 0x8F78
, 0x9188, 0x81A9, 0xB1CA, 0xA1EB, 0xD10C, 0xC12D, 0xF14E, 0xE16F
, 0x1080, 0x00A1, 0x30C2, 0x20E3, 0x5004, 0x4025, 0x7046, 0x6067
, 0x83B9, 0x9398, 0xA3FB, 0xB3DA, 0xC33D, 0xD31C, 0xE37F, 0xF35E
, 0x02B1, 0x1290, 0x22F3, 0x32D2, 0x4235, 0x5214, 0x6277, 0x7256
, 0xB5EA, 0xA5CB, 0x95A8, 0x8589, 0xF56E, 0xE54F, 0xD52C, 0xC50D
, 0x34E2, 0x24C3, 0x14A0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405
, 0xA7DB, 0xB7FA, 0x8799, 0x97B8, 0xE75F, 0xF77E, 0xC71D, 0xD73C
, 0x26D3, 0x36F2, 0x0691, 0x16B0, 0x6657, 0x7676, 0x4615, 0x5634
, 0xD94C, 0xC96D, 0xF90E, 0xE92F, 0x99C8, 0x89E9, 0xB98A, 0xA9AB
, 0x5844, 0x4865, 0x7806, 0x6827, 0x18C0, 0x08E1, 0x3882, 0x28A3
, 0xCB7D, 0xDB5C, 0xEB3F, 0xFB1E, 0x8BF9, 0x9BD8, 0xABBB, 0xBB9A
, 0x4A75, 0x5A54, 0x6A37, 0x7A16, 0x0AF1, 0x1AD0, 0x2AB3, 0x3A92
, 0xFD2E, 0xED0F, 0xDD6C, 0xCD4D, 0xBDAA, 0xAD8B, 0x9DE8, 0x8DC9
, 0x7C26, 0x6C07, 0x5C64, 0x4C45, 0x3CA2, 0x2C83, 0x1CE0, 0x0CC1
, 0xEF1F, 0xFF3E, 0xCF5D, 0xDF7C, 0xAF9B, 0xBFBA, 0x8FD9, 0x9FF8
, 0x6E17, 0x7E36, 0x4E55, 0x5E74, 0x2E93, 0x3EB2, 0x0ED1, 0x1EF0
};

u_int16_t
/*****************************************************************************************************
 *****************************************************************************************************
                                                                                                  ****/
recobe_crccitt(const u_char *octets, ssize_t len) {                                              /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************
 *  u_int16_t r;                                                                                     *
 *  int       i;                                                                                     *
 *                                                                                                   *
 *  for (r = 0; len > 0; --len) {                                                                    *
 *      r = r ^ ((u_int16_t)*octets++ << 8);                                                         *
 *      for (i = 8; i > 0; --i) {                                                                    *
 *          if  (r & 0x8000) {                                                                       *
 *              r = r << 1 ^ 0x1021;                                                                 *
 *          } else {                                                                                 *
 *              r = r << 1;                                                                          *
 *  }   }   }                                                                                        *
 *  return(r);                                                                                       *
 *****************************************************************************************************/
    u_int16_t r;
    ssize_t   i;

    r = 0;
    if  (len < 0) {
        for (i = -len; i > 0;) r = (u_int16_t)(r << 8) ^ crccitt[((r >> 8) ^ octets[--i]) & 0x00FF];
    } else {
        for (i = 0; i < len;) r = (u_int16_t)(r << 8) ^ crccitt[((r >> 8) ^ octets[i++]) & 0x00FF];
    }
    return(r);
}

static int
/*****************************************************************************************************
                                                                                                  ****/
ccitt(recobe_chain *chain, recobe_chunk *chunk) {                                                /****
                                                                                                  ****
 *****************************************************************************************************/
#   define blin_internal_flags (chain->flags)
    ssize_t       intail;
    u_int64_t     wchr;
    int           ex = EX_OK;
    u_int32_t    *r;

    ifBLIN_QX3("+ %08X %"BLIN_X, chain->flags, BLIN_I(chain));
    intail = (ssize_t)recobe_chunksize(chain->tail);
    if  (!(chain->flags & RECOBE_USED)) {
        r = (u_int32_t *)&chain->converdata;
        *r = 0;
        chain->flags |= RECOBE_DOEND | RECOBE_USED;
    } else if (!(chain->flags & RECOBE_DOEND)) {
        chunk->position = RECOBE_EOT;
        goto out;
    }
    r = (u_int32_t *)&chain->converdata;
    if  (intail < 1) {
        *r &= 0x00FFFF;
        if  (0 > (ex = recobe_outword(chain, chunk, *r))) ifBLIN_QW0("out");
        chain->flags &= ~RECOBE_DOEND;
    } else while(0 < recobe_inword(chain, chain->tail, &wchr)) {
        *r = (*r << 8) ^ crccitt[((*r >> 8) ^ wchr) & 0x00FF];
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

u_int8_t
/*****************************************************************************************************
 *****************************************************************************************************
                                                                                                  ****/
recobe_csum(const u_char *octets, ssize_t len) {                                                 /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int32_t r;
    ssize_t   i;

    r = 0;
    if  (0 > len) len = -len;
    for (i = 0; i < len; ++i) r += octets[i];
    return(r & 0x000000FF);
}

static int
/*****************************************************************************************************
                                                                                                  ****/
csum(recobe_chain *chain, recobe_chunk *chunk) {                                                 /****
                                                                                                  ****
 *****************************************************************************************************/
#   define blin_internal_flags (chain->flags)
    ssize_t       intail;
    u_int64_t     wchr;
    int           ex = EX_OK;
    u_int32_t    *r;

    ifBLIN_QX3("+ %08X %"BLIN_X, chain->flags, BLIN_I(chain));
    intail = (ssize_t)recobe_chunksize(chain->tail);
    if  (!(chain->flags & RECOBE_USED)) {
        r = (u_int32_t *)&chain->converdata;
        *r = 0;
        chain->flags |= RECOBE_DOEND | RECOBE_USED;
    } else if (!(chain->flags & RECOBE_DOEND)) {
        chunk->position = RECOBE_EOT;
        goto out;
    }
    r = (u_int32_t *)&chain->converdata;
    if  (intail < 1) {
        *r &= 0x00FF;
        if  (0 > (ex = recobe_outword(chain, chunk, *r))) ifBLIN_QW0("out");
        chain->flags &= ~RECOBE_DOEND;
    } else {
        while(0 < recobe_inword(chain, chain->tail, &wchr)) *r += wchr;
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

u_int8_t
/*****************************************************************************************************
 *****************************************************************************************************
                                                                                                  ****/
recobe_csum1(const u_char *octets, ssize_t len) {                                                /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
    u_int32_t r;
    ssize_t   i;

    r = 0;
    if  (0 > len) len = -len;
    for (i = 0; i < len; ++i) r += octets[i];
    --r;
    return(r & 0x000000FF);
}

static int
/*****************************************************************************************************
                                                                                                  ****/
csum1(recobe_chain *chain, recobe_chunk *chunk) {                                                /****
                                                                                                  ****
 *****************************************************************************************************/
#   define blin_internal_flags (chain->flags)
    ssize_t       intail;
    u_int64_t     wchr;
    int           ex = EX_OK;
    u_int32_t    *r;

    ifBLIN_QX3("+ %08X %"BLIN_X, chain->flags, BLIN_I(chain));
    intail = (ssize_t)recobe_chunksize(chain->tail);
    if  (!(chain->flags & RECOBE_USED)) {
        r = (u_int32_t *)&chain->converdata;
        *r = 0;
        chain->flags |= RECOBE_DOEND | RECOBE_USED;
    } else if (!(chain->flags & RECOBE_DOEND)) {
        chunk->position = RECOBE_EOT;
        goto out;
    }
    r = (u_int32_t *)&chain->converdata;
    if  (intail < 1) {
        --*r;
        *r &= 0x00FF;
        if  (0 > (ex = recobe_outword(chain, chunk, *r))) ifBLIN_QW0("out");
        chain->flags &= ~RECOBE_DOEND;
    } else {
        while(0 < recobe_inword(chain, chain->tail, &wchr)) *r += wchr;
    }
out:
    ifBLIN_QX3("- %d", ex);
    return(ex);
#   undef blin_internal_flags
}

static const recobe_chunker
/*****************************************************************************************************
                                                                                                  ****/
intercode[CRC_L] =                                                                               /****
                                                                                                  ****
 *****************************************************************************************************/
{ NULL
, csum1
, crc24
, crc32
, ccitt
, csum
};

static const u_int16_t
/*****************************************************************************************************
 ****  Output min word count                                                                      ****
 *****************************************************************************************************
                                                                                                  ****/
interoutwlen[CRC_L] =                                                                            /****
                                                                                                  ****
 *****************************************************************************************************/
{ 0
, 1
, 3
, 4
, 2
, 1
};

recobe_chain *
/*****************************************************************************************************
 *****************************************************************************************************
 ****  chain c -k  p.                                                            ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_opens(recobe_conf *cf, const char *p, recobe_chain *next) {                               /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
#   define blin_internal_flags (cf->flags)
    recobe_chain *chain = NULL;

    ifBLIN_QX3("+ w:%s %"BLIN_X, p, BLIN_I(next));
    if  (!(chain = malloc(sizeof(recobe_chain)))) {
        ifBLIN_QW0("malloc(%zd)", sizeof(recobe_chain));
        goto out;
    }
    chain->cnv = convbyname(p);
    if  (!!chain->cnv.nonesrc && !!chain->cnv.nonedst) {
        ifBLIN_QX0("No %s recode", p);
        free(chain);
        chain = NULL;
        errno = ENOENT;
        goto out;
    }
    chain->convertor = intercode[chain->cnv.encodst];
    if  (!chain->convertor) {
        recobe_cnverr(__FILE__, __func__, __LINE__, "No recode", chain->cnv);
        free(chain);
        chain = NULL;
        errno = ENOSYS;
        goto out;
    }
    chain->cf = cf;
    chain->flags = (cf->flags & ~(RECOBE_USED | RECOBE_CHAINER)) | RECOBE_DOEND;
    chain->tail = NULL;
    chain->next = next;
    chain->inwlen = 1;
    chain->outwlen = interoutwlen[chain->cnv.encodst];
    chain->inwnum = 1;
    chain->outwnum = 1;
out:
    RECOBE_CHAIN(chain);
    ifBLIN_QX3("- %"BLIN_X, BLIN_I(chain));
    return(chain);
#   undef blin_internal_flags
}

const recobe_c_tdm
/*****************************************************************************************************
 *****************************************************************************************************
 ****   WIDE                                                                       ****
 *****************************************************************************************************
                                                                                                  ****/
recobe_crc_dm =                                                                                  /****
                                                                                                  ****
 *****************************************************************************************************
 *****************************************************************************************************/
{ CRC_L
, 4
#ifdef __amd64__
, 0
#endif
, "CRC"
, encolist
, codnumbyid
, convbyname
, {NULL}
};
